<?php
// Function to parse .env file
function parseEnv($filePath) {
    if (!file_exists($filePath) || !is_readable($filePath)) {
        die("Error: .env file not found or not readable at $filePath");
    }

    $config = [];
    $lines = file($filePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);

    foreach ($lines as $line) {
        // Skip comments and empty lines
        if (strpos(trim($line), '#') === 0) {
            continue;
        }

        // Split line into key and value
        if (strpos($line, '=') !== false) {
            list($key, $value) = explode('=', $line, 2);
            $key = trim($key);
            $value = trim($value);

            // Remove surrounding quotes if present
            if (preg_match('/^"(.*)"$/', $value, $matches) || preg_match("/^'(.*)'$/", $value, $matches)) {
                $value = $matches[1];
            }

            $config[$key] = $value;
        }
    }

    return $config;
}

// Load .env file from C:\xampp\htdocs\zionist\.env
$envFilePath = 'C:/xampp/htdocs/bungoma/.env';
$env = parseEnv($envFilePath);

// Extract database connection parameters
$servername = $env['DB_HOST'] ?? 'localhost';
$port = $env['DB_PORT'] ?? '3306';
$username = $env['DB_USERNAME'] ?? null;
$password = $env['DB_PASSWORD'] ?? null;
$dbname = $env['DB_DATABASE'] ?? null;

// Validate required parameters
if (!$username || !$dbname) {
    die("Error: Missing required database connection parameters (DB_USERNAME or DB_DATABASE) in .env file.");
}

// Database connection
$conn = new mysqli($servername, $username, $password, $dbname, $port);
if ($conn->connect_error) {
    die("Connection failed: " . $conn->connect_error);
}

// Initialize date range (default to last 30 days)
$start_date = isset($_POST['start_date']) ? $_POST['start_date'] : date('Y-m-d', strtotime('-30 days'));
$end_date = isset($_POST['end_date']) ? $_POST['end_date'] : date('Y-m-d');

// Prepare and execute the SQL query
$sql = "
    SELECT 
        p.id AS item_id,
        p.name AS item_name,
        COALESCE((
            SELECT SUM(pl.quantity) 
            FROM purchase_lines pl 
            WHERE pl.product_id = p.id 
            AND pl.created_at < ? 
        ), 0) - 
        COALESCE((
            SELECT SUM(tsl.quantity) 
            FROM transaction_sell_lines tsl 
            WHERE tsl.product_id = p.id 
            AND tsl.created_at < ? 
        ), 0) + 
        COALESCE((
            SELECT SUM(sal.quantity) 
            FROM stock_adjustment_lines sal 
            WHERE sal.product_id = p.id 
            AND sal.created_at < ? 
        ), 0) AS opening_qty,
        COALESCE((
            SELECT SUM(pl.quantity) 
            FROM purchase_lines pl 
            WHERE pl.product_id = p.id 
            AND pl.created_at BETWEEN ? AND ? 
        ), 0) AS qty_purchased,
        COALESCE((
            SELECT SUM(tsl.quantity) 
            FROM transaction_sell_lines tsl 
            WHERE tsl.product_id = p.id 
            AND tsl.created_at BETWEEN ? AND ? 
        ), 0) AS qty_sold,
        COALESCE((
            SELECT SUM(sal.quantity) 
            FROM stock_adjustment_lines sal 
            WHERE sal.product_id = p.id 
            AND sal.created_at BETWEEN ? AND ? 
        ), 0) AS qty_adjusted,
        (
            COALESCE((
                SELECT SUM(pl.quantity) 
                FROM purchase_lines pl 
                WHERE pl.product_id = p.id 
                AND pl.created_at <= ? 
            ), 0) - 
            COALESCE((
                SELECT SUM(tsl.quantity) 
                FROM transaction_sell_lines tsl 
                WHERE tsl.product_id = p.id 
                AND tsl.created_at <= ? 
            ), 0) + 
            COALESCE((
                SELECT SUM(sal.quantity) 
                FROM stock_adjustment_lines sal 
                WHERE sal.product_id = p.id 
                AND sal.created_at <= ? 
            ), 0)
        ) AS latest_balance
    FROM products p
    WHERE p.business_id = 1
    ORDER BY p.name
";

$stmt = $conn->prepare($sql);
$stmt->bind_param(
    "ssssssssssss",
    $start_date,
    $start_date,
    $start_date,
    $start_date,
    $end_date,
    $start_date,
    $end_date,
    $start_date,
    $end_date,
    $end_date,
    $end_date,
    $end_date
);
$stmt->execute();
$result = $stmt->get_result();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Inventory Report</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            margin: 0;
            padding: 20px;
            background-color: #f4f4f9;
        }
        h1 {
            text-align: center;
            color: #333;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background-color: #fff;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
        }
        .form-container {
            margin-bottom: 20px;
            text-align: center;
        }
        .form-container label {
            margin-right: 10px;
            font-weight: bold;
        }
        .form-container input[type="date"] {
            padding: 8px;
            margin-right: 10px;
            border: 1px solid #ccc;
            border-radius: 4px;
        }
        .form-container button {
            padding: 8px 16px;
            background-color: #007bff;
            color: #fff;
            border: none;
            border-radius: 4px;
            cursor: pointer;
        }
        .form-container button:hover {
            background-color: #0056b3;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }
        th, td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #ddd;
        }
        th {
            background-color: #007bff;
            color: #fff;
        }
        tr:hover {
            background-color: #f5f5f5;
        }
        td {
            text-align: right;
        }
        td:first-child, td:nth-child(2), td:nth-child(3) {
            text-align: left;
        }
        .no-data {
            text-align: center;
            color: #666;
            padding: 20px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>Inventory Report</h1>
        <div class="form-container">
            <form method="POST" action="">
                <label for="start_date">Start Date:</label>
                <input type="date" id="start_date" name="start_date" value="<?php echo htmlspecialchars($start_date); ?>" required>
                <label for="end_date">End Date:</label>
                <input type="date" id="end_date" name="end_date" value="<?php echo htmlspecialchars($end_date); ?>" required>
                <button type="submit">Generate Report</button>
            </form>
        </div>
        <table>
            <thead>
                <tr>
                    <th>#</th>
                    <th>Item ID</th>
                    <th>Item Name</th>
                    <th>Opening Qty</th>
                    <th>Qty Sold</th>
                    <th>Qty Purchased</th>
                    <th>Qty Adjusted</th>
                    <th>Latest Balance</th>
                </tr>
            </thead>
            <tbody>
                <?php if ($result->num_rows > 0): ?>
                    <?php $row_number = 1; ?>
                    <?php while ($row = $result->fetch_assoc()): ?>
                        <tr>
                            <td><?php echo $row_number++; ?></td>
                            <td><?php echo htmlspecialchars($row['item_id']); ?></td>
                            <td><?php echo htmlspecialchars($row['item_name']); ?></td>
                            <td><?php echo number_format($row['opening_qty'], 2); ?></td>
                            <td><?php echo number_format($row['qty_sold'], 2); ?></td>
                            <td><?php echo number_format($row['qty_purchased'], 2); ?></td>
                            <td><?php echo number_format($row['qty_adjusted'], 2); ?></td>
                            <td><?php echo number_format($row['latest_balance'], 2); ?></td>
                        </tr>
                    <?php endwhile; ?>
                <?php else: ?>
                    <tr>
                        <td colspan="8" class="no-data">No data available for the selected date range.</td>
                    </tr>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</body>
</html>

<?php
$stmt->close();
$conn->close();
?>